// SPDX-License-Identifier: GPL-2.0-only
/*
 * RUBIK Pi Fan Control
 *
 * Copyright (C) 2025, Thundercomm All rights reserved.
 *
 * Author: Hongyang Zhao <hongyang.zhao@thundersoft.com>
 */
#include <errno.h>
#include <fcntl.h>
#include <getopt.h>
#include <libgen.h>
#include <limits.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <unistd.h>
#include <syslog.h>
#include <glob.h>
#include <dirent.h>
#include <sys/stat.h>
#include <time.h>

#include <sys/epoll.h>
#include <sys/stat.h>
#include <sys/types.h>

#include <thermal.h>
#include <thermal-tools.h>

struct thermal_data {
	struct thermal_zone *tz;
	struct thermal_handler *th;
};

enum {
	THERMAL_ENGINE_SUCCESS = 0,
	THERMAL_ENGINE_OPTION_perror,
	THERMAL_ENGINE_DAEMON_perror,
	THERMAL_ENGINE_LOG_perror,
	THERMAL_ENGINE_THERMAL_perror,
	THERMAL_ENGINE_MAINLOOP_perror,
	THERMAL_ENGINE_FAN_NOT_FIND_perror,
};

#define MAX_PATH_LENGTH 128

#define LED_PATH_FORMAT "/sys/class/leds/%s/%s"

#define TEMP_THRESHOLD_LOW     65000
#define TEMP_THRESHOLD_MID     80000
#define TEMP_THRESHOLD_HIGH    90000

#define CPU0_TRIP_ID 10

#define FAN_OFF                0
#define FAN_LOW                64
#define FAN_MEDIUM             128
#define FAN_HIGH               255
#define CPU_TEMP_NODE          "/sys/class/thermal/thermal_zone10/temp"

static char fan_node_path[256] = {0};

static const char *boradcast_dir = "/dev/pts";

/** boardcast **/
static void send_to_terminal(const char *device, const char *message)
{
	time_t now;
	char header[100];

	int fd = open(device, O_WRONLY | O_NOCTTY);
	if (fd < 0) return;

	now = time(NULL);
	snprintf(header, sizeof(header), "\n\033[1;31mbroadcast %s\033[0m\n",
			ctime(&now));
	dprintf(fd, "%s\033[0;37m%s\033[0m\n\n", header, message);
	close(fd);
}

static int terminal_broadcast(const char *message)
{
	DIR *dir;
	struct dirent *entry;
	char path[256];
	struct stat st;

	dir = opendir(boradcast_dir);
	if (!dir) {
		syslog(LOG_ERR, "can not open %s: %m", boradcast_dir);
		return 1;
	}

	while ((entry = readdir(dir)) != NULL) {
		if (strcmp(entry->d_name, ".") == 0 ||
			strcmp(entry->d_name, "..") == 0) continue;

		snprintf(path, sizeof(path), "%s/%s", boradcast_dir, entry->d_name);

		if (stat(path, &st)) continue;
		if (!S_ISCHR(st.st_mode)) continue;

		send_to_terminal(path, message);
	}

	send_to_terminal("/dev/ttyMSM0", message);

	closedir(dir);
	return 0;
}
/** boardcast end **/

/** fan **/
static int set_fan_speed(int level)
{
	FILE *fp = fopen(fan_node_path, "w");
	if (!fp) {
		syslog(LOG_ERR, "Unable to turn on the fan");
		return -1;
	}

	if (fprintf(fp, "%d\n", level) < 0) {
		syslog(LOG_ERR, "Failed to set fan speed");
		fclose(fp);
		return -1;
	}

	fclose(fp);

	syslog(LOG_DEBUG, "The fan speed is set to %d", level);

	return 0;
}

static int get_fan_speed()
{
	int speed;
	FILE *fp;

	fp = fopen(fan_node_path, "r");
	if (!fp) {
		syslog(LOG_ERR, "Unable to turn on the fan");
		return -1;
	}

	if (fscanf(fp, "%d\n", &speed) < 0) {
		syslog(LOG_ERR, "Failed to get fan speed");
		fclose(fp);
		return -1;
	}

	fclose(fp);

	syslog(LOG_DEBUG, "The fan speed is %d", speed);

	return speed;
}

static int fan_stop()
{
	set_fan_speed(FAN_OFF);
	return 0;
}

static int init_fan_node_glob()
{
	glob_t globbuf;
	const char *pattern = "/sys/devices/platform/pwm-fan/hwmon/hwmon*/pwm1";

	if (glob(pattern, 0, NULL, &globbuf) == 0) {
		if (globbuf.gl_pathc > 0) {
			/* Find the fan node */
			strncpy(fan_node_path, globbuf.gl_pathv[0], sizeof(fan_node_path));
			globfree(&globbuf);
			return 0;
		}
	}

	globfree(&globbuf);

	return -1;
}
/** fan end **/

/** led **/
static int write_led_value(const char *color, const char *file, const char *value)
{
	char path[MAX_PATH_LENGTH];
	int ret=0;
	FILE *fp;

	snprintf(path, sizeof(path), LED_PATH_FORMAT, color, file);

	fp = fopen(path, "w");
	if (!fp) {
		syslog(LOG_ERR, "Unable to open LED %s for %s: %m", color, file);
		return -1;
	}

	if (fprintf(fp, "%s\n", value) < 0) {
		syslog(LOG_ERR, "Failed to write '%s' to %s: %m", value, path);
		ret = -1;
	}

	fclose(fp);
	return ret;
}

static int set_led_trigger(const char *color, const char *trigger)
{
	return write_led_value(color, "trigger", trigger);
}

static int set_led_brightness(const char *color, int brightness)
{
	char value[16];

	snprintf(value, sizeof(value), "%d", brightness);
	return write_led_value(color, "brightness", value);
}

static int heartbeat_color_set(const char *color, int brightness)
{
	const char *leds[] = {"green", "red", "blue"};

	for (size_t i = 0; i < sizeof(leds)/sizeof(leds[0]); i++) {
		if (set_led_trigger(leds[i], "none") < 0) {
			return -1;
		}
	}

	if (set_led_trigger(color, "heartbeat") < 0) {
		return -1;
	}

	if (set_led_brightness(color, brightness) < 0) {
		return -1;
	}

	syslog(LOG_DEBUG, "Heartbeat color set to blue");
	return 0;
}
/** led end **/

/** cpu **/
static int get_cpu_temp()
{
	int temp;

	FILE *fp = fopen(CPU_TEMP_NODE, "r");
	if (!fp) {
		syslog(LOG_ERR, "Unable to open CPU Temperature node");
		return -1;
	}

	if (fscanf(fp, "%d\n", &temp) < 0) {
		syslog(LOG_ERR, "Failed to get cpu temp");
		fclose(fp);
		return -1;
	}

	syslog(LOG_DEBUG, "The cpu temp is %d", temp);

	fclose(fp);

	return temp;
}
/** cpu end **/

/** thermal **/
static int trip_high_cb(int tz_id, int trip_id, int temp, void *arg)
{
	syslog(LOG_INFO, "High temperature trigger: zone=%d, trip= %d, temp=%d\n",
				tz_id, trip_id, temp);

	if (temp >= TEMP_THRESHOLD_HIGH) {
		set_fan_speed(FAN_HIGH);
		if ( trip_id == CPU0_TRIP_ID) {
			heartbeat_color_set("blue", 10);
			terminal_broadcast("Overtemperature detected! Please check fan status.");
		}
	}
	else if (temp >= TEMP_THRESHOLD_MID) {
		set_fan_speed(FAN_MEDIUM);
	}
	else if (temp >= TEMP_THRESHOLD_LOW) {
		set_fan_speed(FAN_LOW);
	}

	return 0;
}

static int trip_low_cb(int tz_id, int trip_id, int temp, void *arg)
{
	syslog(LOG_INFO, "Low temperature trigger: zone=%d, trip= %d, temp=%d\n",
				tz_id, trip_id, temp);

	if (temp < TEMP_THRESHOLD_LOW) {
		set_fan_speed(FAN_OFF);
		heartbeat_color_set("green", 5);
	}
	else if (temp < TEMP_THRESHOLD_MID) {
		set_fan_speed(FAN_LOW);
	} else if (temp < TEMP_THRESHOLD_HIGH) {
		set_fan_speed(FAN_MEDIUM);
	}

	return 0;
}

static int thermal_event(int fd, void *arg)
{
	struct thermal_data *td = arg;

	return thermal_events_handle(td->th, td);
}

static int fan_cpu_temp()
{
	int temp;

	temp = get_cpu_temp();
	if (temp < 0) {
		syslog(LOG_ERR, "Failed to get CPU temperature");
		return -1;
	}

	if (temp >= TEMP_THRESHOLD_HIGH) {
		set_fan_speed(FAN_HIGH);
		heartbeat_color_set("blue", 10);
		terminal_broadcast("Overtemperature detected! Please check fan status.");
	}
	else if (temp >= TEMP_THRESHOLD_MID) {
		set_fan_speed(FAN_MEDIUM);
		heartbeat_color_set("green", 5);
	}
	else if (temp >= TEMP_THRESHOLD_LOW) {
		set_fan_speed(FAN_LOW);
		heartbeat_color_set("green", 5);
	}
	else {
		set_fan_speed(FAN_OFF);
		heartbeat_color_set("green", 5);
	}

	return 0;
}

static struct thermal_ops ops = {
	.events.trip_high	= trip_high_cb,
	.events.trip_low	= trip_low_cb,
};

struct thermal_data td;

/** thermal end **/

/** log **/
static int thermal_log_init()
{
	openlog("rubikpi-thermal", LOG_PID | LOG_CONS, LOG_DAEMON);

	return 0;
}
/** log end **/

int main(int argc, char **argv)
{
	int ret;

	thermal_log_init();

	ret = init_fan_node_glob();
	if (ret) {
		return THERMAL_ENGINE_FAN_NOT_FIND_perror;
	}

	fan_cpu_temp();

	td.th = thermal_init(&ops);
	if (!td.th) {
		syslog(LOG_ERR, "Failed to initialize the thermal library\n");
		return THERMAL_ENGINE_THERMAL_perror;
	}

	if (mainloop_init()) {
		syslog(LOG_ERR, "Failed to initialize the mainloop\n");
		return THERMAL_ENGINE_MAINLOOP_perror;
	}

	if (mainloop_add(thermal_events_fd(td.th), thermal_event, &td)) {
		syslog(LOG_ERR, "Failed to setup the mainloop\n");
		return THERMAL_ENGINE_MAINLOOP_perror;
	}

	if (mainloop(-1)) {
		syslog(LOG_ERR, "Mainloop failed\n");
		return THERMAL_ENGINE_MAINLOOP_perror;
	}

	return THERMAL_ENGINE_SUCCESS;
}
